package ga.view.billard.nodes;

import ga.core.individual.IAgeIndividual;
import ga.view.billard.BillardEvaluationState;
import ga.view.factory.SceneFactory;

import com.jme3.asset.AssetManager;
import com.jme3.bullet.control.RigidBodyControl;
import com.jme3.math.ColorRGBA;
import com.jme3.math.Vector3f;
import com.jme3.renderer.queue.RenderQueue.ShadowMode;
import com.jme3.scene.Geometry;
import com.jme3.scene.Node;

/**
 * Object node the for {@link BillardEvaluationState}.
 * 
 * @param <T>
 *          The generic type of the individuals.
 * 
 * @since 12.08.2012
 * @author Stephan Dreyer
 */
public class BillardObjectNode<T extends IAgeIndividual<T>> extends Node {
  private final T individual;

  private final Geometry geometry;

  private float colorVal;

  private float relSize = 1f;

  private float alpha = 1f;

  private boolean fadeOut;
  private boolean fadeIn;

  private final RigidBodyControl control;

  /**
   * Instantiates a new billard object node.
   * 
   * @param assetManager
   *          the asset manager
   * @param s
   *          the s
   * @param individual
   *          the individual
   * 
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public BillardObjectNode(final AssetManager assetManager, final Geometry s,
      final T individual) {
    geometry = s;
    this.individual = individual;

    setShadowMode(ShadowMode.Cast);

    final Vector3f translation = geometry.getLocalTranslation();

    geometry.setLocalTranslation(Vector3f.ZERO);

    attachChild(geometry);
    // control.setCollisionShape();
    setLocalTranslation(translation);

    control = new RigidBodyControl(SceneFactory.getCollisionShape(geometry),
        10f);
    addControl(control);

    // control.setMass(10f);
    control.setDamping(.5f, .5f);
    control.setFriction(0.1f);

    control.setSleepingThresholds(3f, 3f);
    control.setRestitution(.001f);

  }

  @Override
  public void updateLogicalState(final float tpf) {
    super.updateLogicalState(tpf);

    if (colorVal > 0f) {
      colorVal -= tpf * 1f;
      colorVal = Math.min(1f, colorVal);

      geometry.getMaterial().setColor("Ambient",
          new ColorRGBA(colorVal, colorVal, colorVal, 1f));

    }

    if (fadeOut && relSize > 0f) {
      final ColorRGBA col = (ColorRGBA) geometry.getMaterial()
          .getParam("Diffuse").getValue();

      geometry.getMaterial().setColor("Diffuse",
          col.set(col.r, col.b, col.b, alpha));

      alpha -= tpf * 2f;
      alpha = Math.max(0f, alpha);

      geometry.setLocalScale(geometry.getLocalScale().mult(relSize));
      relSize -= tpf * .2f;
    } else if (fadeIn && alpha < 1f) {

      alpha += tpf * 2f;
      alpha = Math.min(1f, alpha);

      final ColorRGBA col = (ColorRGBA) geometry.getMaterial()
          .getParam("Diffuse").getValue();

      geometry.getMaterial().setColor("Diffuse",
          col.set(col.r, col.b, col.b, alpha));
    }
  }

  @Override
  protected void setParent(final Node parent) {
    super.setParent(parent);
    control.setPhysicsLocation(this.getLocalTranslation());
  }

  /**
   * Indicates that the node has been selected.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void doSelect() {
    colorVal = 1f;
  }

  /**
   * Indicates that the node has beeing destroyed.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void doFadeOut() {
    fadeOut = true;
  }

  /**
   * Indicates that the node has been created.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void doFadeIn() {
    fadeIn = true;
    alpha = 0.1f;
  }

  /**
   * Resizes the node depending on the age.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void updateSizeByAge() {
    final float newScale = 1f - (.5f * individual.getAgeNormalized());
    geometry.setLocalScale(newScale);

    control.getCollisionShape().setScale(
        geometry.getWorldTransform().getScale());
    // setCollisionShape(SceneFactory.getCollisionShape(geometry));
  }

  /**
   * Checks if the node should be removed.
   * 
   * @return <code>true</code> if the node should be removed.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public boolean mustRemove() {
    return relSize < 0;
  }

  /**
   * Gets the individual.
   * 
   * @return the individual
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public T getIndividual() {
    return individual;
  }

  /**
   * Gets the geometry.
   * 
   * @return the geometry
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public Geometry getGeometry() {
    return geometry;
  }

  @Override
  public String toString() {
    return getClass().getSimpleName() + " " + individual.toString();
  }
}
